﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.Utils;
using System.IO;
using BMS.ServicesWrapper.Security;
using BMS.Facade.Data;

namespace BMS.Facade
{
    public class ReportUtil
    {

        /// <summary>
        /// Encodes parameters to report server
        /// </summary>
        /// <param name="value">parameter object</param>
        /// <returns>encoded string to be sent over query string</returns>
        public static string Encrypt(ReportUtil value)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                StringBuilder sb = new StringBuilder();
                sb.AppendLine(value.Domain);
                sb.AppendLine(value.Password);
                sb.AppendLine(value.ReportName);
                sb.AppendLine(value.ReportPath);
                sb.AppendLine(value.ReportServerUrl);
                sb.AppendLine(value.UserId);
                sb.AppendLine(value.UserName);

                foreach (var key in value.Parameters.Keys)
                {
                    sb.AppendLine(key);
                    sb.AppendLine(value.Parameters[key]);
                }
                User user = FacadeManager.UserInterface.GetProfile();
                return CustomEncryption.Encrypt(sb.ToString(), user.Salt);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Decodes parameters to report server
        /// </summary>
        /// <param name="encryptedValue">encoded string from query string to be interpreted by de ReportViewer</param>
        /// <returns>parameters object</returns>
        public static ReportUtil Decrypt(String encryptedValue)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                User user = FacadeManager.UserInterface.GetProfile();
                String decryptedQuery = CustomEncryption.Decrypt(encryptedValue, user.Salt);
                using (StringReader sr = new StringReader(decryptedQuery))
                {
                    ReportUtil rg = new ReportUtil();
                    rg.Domain = sr.ReadLine();
                    rg.Password = sr.ReadLine();
                    rg.ReportName = sr.ReadLine();
                    rg.ReportPath = sr.ReadLine();
                    rg.ReportServerUrl = sr.ReadLine();
                    rg.UserId = sr.ReadLine();
                    rg.UserName = sr.ReadLine();

                    rg._parameters = new Dictionary<string, string>();
                    string key = null;
                    string val = null;
                    do
                    {
                        key = sr.ReadLine();
                        if (key == null)
                            break;
                        val = sr.ReadLine();
                        rg.Parameters[key] = val;
                    }
                    while (key != null);

                    return rg;
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        string _reportServerUrl = String.Empty;
        string _reportPath = String.Empty;
        string _reportName = String.Empty;
        string _userName = String.Empty;
        string _domain = String.Empty;
        string _password = String.Empty;
        string _userId = String.Empty;
        Dictionary<string, string> _parameters = new Dictionary<string, string>();

        /// <summary>
        /// Parameter collection to be sent to a report
        /// </summary>
        public Dictionary<string, string> Parameters
        {
            get { return _parameters; }
        }

        /// <summary>
        /// User domain
        /// </summary>
        public string Domain
        {
            get { return _domain; }
            set { _domain = value; }
        }
        /// <summary>
        /// user password
        /// </summary>
        public string Password
        {
            get { return _password; }
            set { _password = value; }
        }

        /// <summary>
        /// user id
        /// </summary>
        public string UserId
        {
            get { return _userId; }
            set { _userId = value; }
        }

        /// <summary>
        /// Report name taken from the RS service
        /// </summary>
        public string ReportName
        {
            get { return _reportName; }
            set { _reportName = value; }
        }

        /// <summary>
        /// Report path taken from the RS service and used to access the desired report
        /// </summary>
        public string ReportPath
        {
            get { return _reportPath; }
            set { _reportPath = value; }
        }

        /// <summary>
        /// User name requesting the report
        /// </summary>
        public string UserName
        {
            get { return _userName; }
            set { _userName = value; }
        }

        /// <summary>
        /// Report server url
        /// </summary>
        public string ReportServerUrl
        {
            get { return _reportServerUrl; }
            set { _reportServerUrl = value; }
        }

        static string _defaultServerUrl, _defaultDomain, _defaultPassword, _defaultUserName;
        public static void SetDefaultProperties(string serverUrl, string domain, string password, string userName)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                _defaultDomain = domain;
                _defaultPassword = password;
                _defaultServerUrl = serverUrl;
                _defaultUserName = userName;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Builds default ReportUtil instance
        /// </summary>
        /// <param name="fullUserName">username@domain</param>
        /// <returns></returns>
        public static ReportUtil CreateDefaultInstance(string fullUserName)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                ReportUtil reportUtil = new ReportUtil()
                {
                    Domain = _defaultDomain,
                    Password = _defaultPassword,
                    ReportServerUrl = _defaultServerUrl,
                    UserName = _defaultUserName,
                    UserId = fullUserName
                };

                return reportUtil;
                
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static string GetURL(IReportInfo reportInfo, string fullUserName)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                ReportUtil reportUtil = CreateDefaultInstance(fullUserName);
                reportUtil.ReportName = reportInfo.Name;
                reportUtil.ReportPath = reportInfo.Path;
                reportUtil.Parameters.Clear();
                if (reportInfo.Parameters != null)
                    foreach (KeyValuePair<String, String> kvp in reportInfo.Parameters)
                        reportUtil.Parameters.Add(kvp.Key, kvp.Value);

                return ReportUtil.Encrypt(reportUtil);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
    }


}
